#!/usr/bin/env sh

JQ=${JQ:=jq}

# Check for packages
JQ_VERSION=$("${JQ}" --version 2>/dev/null)
if [ "${JQ_VERSION}" = 'jq-1.3' ] || [ "${JQ_VERSION}" = 'jq-1.4' ] || [ "${JQ_VERSION}" = 'jq-1.5' ] || [ "${JQ_VERSION}" = 'jq-1.6' ]; then
	echo 'ERROR: installed jq version is too old; must be at least 1.7'
	exit 1
elif [ -z ${JQ_VERSION} ]; then
	echo 'ERROR: please install jq 1.7+'
	exit 1
fi

NICKLE_LOCATION=$(which nickle)
CURL_LOCATION=$(which curl)

if [ ! -f "${NICKLE_LOCATION}" ]; then
	echo "ERROR: please install Nickle"
	exit 1
fi

if [ ! -f "${CURL_LOCATION}" ]; then
	echo "ERROR: please install cURL"
	exit 1
fi

# Only accepts 1 argument: the Minecraft version to download
LOOOKING_FOR=
if [ -z "${1}" ]; then
	echo "usage: ${0} <version>"
	exit 1
else
	LOOKING_FOR="${1}"
fi

PLATFORM=$(uname -s | tr '[:upper:]' '[:lower:]')

MANIFEST_LIST_URL='https://launchermeta.mojang.com/mc/game/version_manifest.json'

mkdir -p assets/indexes

echo 'Retrieving version manifest...'
VERSION_MANIFEST_URL=$(curl "${MANIFEST_LIST_URL}" 2>/dev/null | "${JQ}" -r "(.versions.[] | select(.id==\"${LOOKING_FOR}\")).url")

if [ -z "${VERSION_MANIFEST_URL}" ]; then
	echo "ERROR: Unable to find requested version"
	exit 1
fi

mkdir -p manifests
VERSION_MANIFEST=$(curl ${VERSION_MANIFEST_URL} 2>/dev/null | sed -e 's/\\//g' | tee "manifests/${LOOKING_FOR}.json") # Some versions contain illegal escape sequences which will crash some shells such as Ksh; double-escape it just to be safe

# 1.12 and earlier, natives and regular libraries are contained in different elements. For 1.13 and later, an element may contain both.
echo 'Downloading libraries...'
for I in $(echo "${VERSION_MANIFEST}" | "${JQ}" -c .libraries.[]); do
	if [ "$(echo "${I}" | "${JQ}" .downloads.artifact)" != 'null' ]; then
		echo "Library '$(echo ${I} | "${JQ}" -r .name )' found - downloading..."
		mkdir -p libraries/$(dirname $(echo ${I} | "${JQ}" -r .downloads.artifact.path))
		curl $(echo ${I} | "${JQ}" -r .downloads.artifact.url) > libraries/$(echo "${I}" | "${JQ}" -r .downloads.artifact.path) 2>/dev/null
	fi
	if [ "$(echo "${I}" | "${JQ}" .natives)" != 'null' ]; then
		echo "Downloading native for platform '${PLATFORM}'..."
		if [ "$(echo ${I} | "${JQ}" -r ".downloads.classifiers.\"natives-${PLATFORM}\"")" = 'null' ]; then
			echo "WARNING: Mojang does not distribute a '$(echo ${I} | "${JQ}" -r .name)' native for this platform. This is probably not an issue unless you are using an obscure platform."
		else
			mkdir -p libraries/$(dirname $(echo ${I} | "${JQ}" -r ".downloads.classifiers.\"natives-${PLATFORM}\".path"))
			curl $(echo ${I} | "${JQ}" -r ".downloads.classifiers.\"natives-${PLATFORM}\".url") > libraries/$(echo "${I}" | "${JQ}" -r ".downloads.classifiers.\"natives-${PLATFORM}\".path") 2>/dev/null
		fi
	fi
done

echo 'Retrieving asset index...'
ASSET_INDEX=$(echo "${VERSION_MANIFEST}" | "${JQ}" .assetIndex)

echo 'Retrieving object list...'
OBJECT_LIST=$(curl $(echo ${ASSET_INDEX} | "${JQ}" -r .url) 2>/dev/null | tee assets/indexes/$(echo ${ASSET_INDEX} | "${JQ}" -r .id).json)

HASH_COUNT=$(echo ${OBJECT_LIST} | "${JQ}" '.objects | length')

echo 'Downloading required assets. This may take a few minutes.'
MCRESOURCES='https://resources.download.minecraft.net/'
if [ "$(echo "${ASSET_INDEX}" | "${JQ}" -r .id)" = 'pre-1.6' ] || [ "$(echo "${ASSET_INDEX}" | "${JQ}" -r .id)" = 'legacy' ]; then # Release 1.6 and prior use a different asset layout
	echo "${OBJECT_LIST}" | "${JQ}" -r '.objects | keys[]' | while read -r LINE; do
		H=$(echo ${OBJECT_LIST} | "${JQ}" -r ".objects.[\"${LINE}\"].hash")
		DIR="$(echo "${H}" | cut -c1-2)/"

		mkdir -p "$(dirname "assets/virtual/legacy/${LINE}")"
		curl "${MCRESOURCES}${DIR}${H}" > "assets/virtual/legacy/${LINE}" 2>/dev/null
		echo "Downloaded: assets/virtual/legacy/${LINE}"
	done
else
	IND=0
	for I in $(echo "${OBJECT_LIST}" | "${JQ}" -r .objects.[].hash); do
		IND=$(echo "${IND}" + 1 | nickle)
		DIR="$(echo "${I}" | cut -c1-2)/"

		mkdir -p "assets/objects/${DIR}"

		# If file exists, check its integrity with hash and skip if it's good
		if [ -f assets/objects/${DIR}${I} ] && [ "$(cat "assets/objects/${DIR}${I}" | sha1sum | sed -e 's/  (stdin)//')" = "${I}" ]; then # Removing (stdin) is not needed for most sha1sum implementations
				echo "${IND}/${HASH_COUNT} (${DIR}${I}) is already downloaded; skipping. ($(echo "printf(\"%5l\", ${IND} / ${HASH_COUNT} * 100)" | nickle)%)"
			continue;
		fi

		curl "${MCRESOURCES}${DIR}${I}" > "assets/objects/${DIR}${I}" 2>/dev/null
		echo "Downloading ${IND}/${HASH_COUNT}: ${MCRESOURCES}${DIR}${I} ($(echo "printf(\"%5l\", ${IND} / ${HASH_COUNT} * 100)" | nickle)%)"
	done
fi

echo 'Downloading minecraft jar...'
CLIENT_URL=$(echo "${VERSION_MANIFEST}" | "${JQ}" -r .downloads.client.url)
mkdir -p "libraries/com/mojang/minecraft/${LOOKING_FOR}"
curl "${CLIENT_URL}" > "libraries/com/mojang/minecraft/${LOOKING_FOR}/minecraft-${LOOKING_FOR}-client.jar" 2>/dev/null

echo 'Finished!'
